/*
  Copyright (C) 2000-2008

  Code contributed by Greg Collecutt, Joseph Hope and Paul Cochrane

  This file is part of xmds.

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/*
  $Id: xmds_integrate_ip.cc 1885 2008-03-18 15:24:56Z paultcochrane $
*/

/*! @file xmds_integrate_ip.cc
  @brief Integrate element parsing classes and methods; interaction picture

  More detailed explanation...
*/

#include <xmds_common.h>
#include <xmds_integrate_ip.h>
#include <xmds_simulation.h>
#include <xmds_vector.h>

// **************************************************************************
// **************************************************************************
//                              xmdsIntegrateIP public
// **************************************************************************
// **************************************************************************

extern bool debugFlag;

long nxmdsIntegrateIPs=0;  //!< Number of integrate IP objects

// **************************************************************************
xmdsIntegrateIP::xmdsIntegrateIP(
                                 const xmdsSimulation *const yourSimulation,
                                 const bool& yourVerboseMode) :
  xmdsIntegrate(yourSimulation, yourVerboseMode, false, false) {
  if (debugFlag) {
    nxmdsIntegrateIPs++;
    printf("xmdsIntegrateIP::xmdsIntegrateIP\n");
    printf("nxmdsIntegrateIPs=%li\n", nxmdsIntegrateIPs);
  }
}

// **************************************************************************
xmdsIntegrateIP::~xmdsIntegrateIP() {
  if (debugFlag) {
    nxmdsIntegrateIPs--;
    printf("xmdsIntegrateIP::~xmdsIntegrateIP\n");
    printf("nxmdsIntegrateIPs=%li\n", nxmdsIntegrateIPs);
  }
}

// **************************************************************************
void xmdsIntegrateIP::processElement(
                                     const Element *const yourElement) {
  if (debugFlag) {
    printf("xmdsIntegrateIP::processElement\n");
  }

  // ************************************
  // parse code for operators

  const xmdsVector* mainVector;

  if (!simulation()->field()->getVector("main", mainVector)) {
    throw xmdsException("Internal error in xmdsIntegrateIP::processElement: cannot find 'main' vector");
  }

  XMLString* theCode = propagationCode();

  XMLString nextOperatorName;
  XMLString nextComponentName;
  unsigned long start=0;
  unsigned long end=0;

  while (end<theCode->length()) {
	  if (findNextcoPair(nextOperatorName, nextComponentName, start, end)) {
		  
		  unsigned long nextComponentNumber;
		  
		  if (!mainVector->getComponent(nextComponentName, nextComponentNumber)) {
			  sprintf(errorMessage(), "[%s] is not a component of the main vector", nextComponentName.c_str());
			  throw xmdsException(yourElement, errorMessage());
		  }
		  
		  unsigned long nextOperatorNumber;
		  
		  if (!getKOperator(nextOperatorName, nextOperatorNumber)) {
			  sprintf(errorMessage(), "'%s' was not defined in <k_operators>", nextOperatorName.c_str());
			  throw xmdsException(yourElement, errorMessage());
		  }
		  
		  
		  unsigned long dummycoKey;
		  
		  if (getcoKey(nextComponentNumber, nextOperatorNumber, dummycoKey)) {
			  if (mainVector->componentLength(nextComponentNumber)<2) {
				  sprintf(errorMessage(), "'%s[%s]' used multiply", nextOperatorName.c_str(), nextComponentName.c_str());
				  throw xmdsException(yourElement, errorMessage());
			  }
		  }
		  else {
			  if (verbose()) {
				  printf("adding operator-component pair: %s[%s]\n", nextOperatorName.c_str(), nextComponentName.c_str());
			  }
			  
			  unsigned long dummyLong = mainVector->componentLength(nextComponentNumber);
			  addcoPair(nextComponentNumber, nextOperatorNumber,dummyLong);
		  }
		  
		  if(propagationCode()->data(end)=='(') {
			  theCode->replaceData(start, end-start, "0*");
		  }
		  else {
			  theCode->replaceData(start, end-start, "0");
		  }
		  
		  start = start+1;
	  }
    else {
      start = end;
    }
  }

}

// **************************************************************************
// **************************************************************************
//                              xmdsIntegrateIP protected
// **************************************************************************
// *****************************************************************************
// **************************************************************************
void xmdsIntegrateIP::writePrototypes(
                                      FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateIP::writePrototypes\n");
  }

  if (usesKOperators()) {
    if (AdaptiveIP()||Is9thOrderIP()){

      if (Smallmemory()){
        fprintf(outfile, "void _segment%li_k_propagate(\n", segmentNumber);
        fprintf(outfile, " const double& _step);\n");
        fprintf(outfile, "\n");
      }
      else{
        fprintf(outfile, "void _segment%li_calculate_k_operator_field(double _step);\n", segmentNumber);
        fprintf(outfile, "\n");
        fprintf(outfile, "void _segment%li_k_propagate(int _exponent);\n", segmentNumber);
        fprintf(outfile, "\n");
      }

    }
    else{
      if (constantK()) {
        fprintf(outfile, "void _segment%li_calculate_k_operator_field();\n", segmentNumber);
        fprintf(outfile, "\n");
        fprintf(outfile, "void _segment%li_k_propagate();\n", segmentNumber);
        fprintf(outfile, "\n");
      }
      else {
        fprintf(outfile, "void _segment%li_k_propagate(\n", segmentNumber);
        fprintf(outfile, " const double& _step);\n");
        fprintf(outfile, "\n");
      }
    }
  }
}

// **************************************************************************
void xmdsIntegrateIP::writeRoutines(
                                    FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateIP::writeRoutines\n");
  }

  if (usesKOperators()) {
    if (Is9thOrderIP()){
      if (Smallmemory())
        writeTimeDepkPropagateRoutine(outfile);  // use conventional routine
      else{
        if (constantK())                     // optimized for speed, but needs lots of memory
          writeRK9CalculatekOperatorFieldRoutine(outfile);
        else                         // optimized for speed, but needs even more memory
          writeRK9TimeDepkCalculatekOperatorFieldRoutine(outfile);
        writeRK9kPropagateRoutine(outfile);
      }
    }
    else {
      if (AdaptiveIP()){
        if (Smallmemory())
          writeTimeDepkPropagateRoutine(outfile);  // use conventional routine
        else{
          if (constantK())                     // optimized for speed, but needs lots of memory
            writeARK45CalculatekOperatorFieldRoutine(outfile);
          else                         // optimized for speed, but needs even more memory
            writeARK45TimeDepkCalculatekOperatorFieldRoutine(outfile);
          writeARK45kPropagateRoutine(outfile);
        }
      }
      else{
        if (constantK()) {
          writeCalculatekOperatorFieldRoutine(outfile);
          writeConstkPropagateRoutine(outfile);
        }
        else
          writeTimeDepkPropagateRoutine(outfile);
      }
    }
  }
}

// **************************************************************************
// **************************************************************************
//                              xmdsIntegrateIP private
// **************************************************************************
// **************************************************************************

// **************************************************************************
void xmdsIntegrateIP::writeCalculatekOperatorFieldRoutine(
                                                          FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateIP::writeCalculatekOperatorFieldRoutine\n");
  }

  const unsigned long nDims = simulation()->field()->geometry()->nDims();
  const unsigned long fullSpace = simulation()->field()->geometry()->fullSpace();

  unsigned long i;

  fprintf(outfile, "// *************************\n");
  fprintf(outfile, "void _segment%li_calculate_k_operator_field() {\n", segmentNumber);
  fprintf(outfile, "\n");

  for (i=0; i<nKOperators(); i++) {
    fprintf(outfile, "complex %s;\n", KOperator(i)->c_str());
  }
  fprintf(outfile, "\n");

  fprintf(outfile, "double _step=(%s/(double)%li)/2;\n", interval()->c_str(), lattice());
  fprintf(outfile, "\n");

  if (simulation()->parameters()->errorCheck) {
    fprintf(outfile, "if (_half_step)\n");
    fprintf(outfile, "  _step=_step/2;\n");
    fprintf(outfile, "\n");
  }

  simulation()->field()->vectors2space(outfile, fullSpace, *KVectorNamesList(), "");

  fprintf(outfile, "unsigned long _k_operator_index_pointer=0;\n");

  // Strictly, this is parallelisable, however fixing this would require creating a vector
  simulation()->field()->openLoops(outfile, fullSpace, *KVectorNamesList(), DO_NOT_PARALLELISE_LOOP);

  char indent[64];
  for (i=0; i<nDims; i++) {
    indent[i]=0x09;
  }
  indent[nDims]=0;

  fprintf(outfile, "\n");
  fprintf(outfile, "// ********** Code from k_operators *************\n");
  fprintf(outfile, "%s\n", KOperatorsCode()->c_str());
  fprintf(outfile, "// **********************************************\n");
  fprintf(outfile, "\n");

  for (i=0; i<nKOperators(); i++) {
    fprintf(outfile, "%s_segment%li_k_operator_field[_k_operator_index_pointer + %li].re = exp(%s.re*_step)*cos(%s.im*_step);\n",
            indent, segmentNumber, i, KOperator(i)->c_str(), KOperator(i)->c_str());
    fprintf(outfile, "%s_segment%li_k_operator_field[_k_operator_index_pointer + %li].im = exp(%s.re*_step)*sin(%s.im*_step);\n",
            indent, segmentNumber, i, KOperator(i)->c_str(), KOperator(i)->c_str());
  }
  fprintf(outfile, "\n");
  fprintf(outfile, "%s_k_operator_index_pointer += _segment%li_nkoperators;\n", indent, segmentNumber);
  simulation()->field()->closeLoops(outfile, fullSpace, *KVectorNamesList());

  fprintf(outfile, "}\n");
  fprintf(outfile, "\n");
}


// **************************************************************************
void xmdsIntegrateIP::writeConstkPropagateRoutine(
                                                  FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateIP::writeConstkPropagateRoutine\n");
  }

  const unsigned long fullSpace = simulation()->field()->geometry()->fullSpace();
  const char *const fieldName = simulation()->field()->name()->c_str();

  list<XMLString> tempVectorNamesList;
  tempVectorNamesList.push_back("main");

  const xmdsVector* mainVector;

  if (!simulation()->field()->getVector("main", mainVector)) {
    throw xmdsException("Internal error in xmdsIntegrateIP::writeConstkPropagateRoutine: cannot find 'main' vector");
  }

  fprintf(outfile, "// *************************\n");
  fprintf(outfile, "void _segment%li_k_propagate() {\n", segmentNumber);
  fprintf(outfile, "\n");
  fprintf(outfile, "double _temp;\n");
  fprintf(outfile, "unsigned long _segment%li_kop_index_pointer=0;\n", segmentNumber);
  fprintf(outfile, "unsigned long _active_%s_index_pointer=0;\n", fieldName);
  fprintf(outfile, "\n");

  simulation()->field()->vectors2space(outfile, fullSpace, tempVectorNamesList, "");

  if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
    fprintf(outfile, "for (long _i0=0; _i0<total_local_size; _i0++) {\n");
  }
  else {
    if (simulation()->parameters()->useOpenMP) {
      fprintf(outfile, "#ifdef _OPENMP\n"
              "bool _firstTime = true;\n"
              "#pragma omp parallel for firstprivate(_firstTime, _temp, _segment%li_kop_index_pointer, _active_%s_index_pointer)\n"
              "#endif\n",
              segmentNumber, fieldName);
    }
    fprintf(outfile, "for (long _i0=0; _i0<_%s_size; _i0++) {\n", fieldName);
    if (simulation()->parameters()->useOpenMP) {
      fprintf(outfile, "#ifdef _OPENMP\n"
              "if (_firstTime == true) {\n"
              "\t _firstTime = false;\n"
              "\t _segment%1$li_kop_index_pointer += _i0*_segment%1$li_nkoperators;\n"
              "\t _active_%2$s_index_pointer += _i0*_%2$s_main_ncomponents;\n"
              "}\n"
              "#endif // _OPENMP\n\n",
              segmentNumber, fieldName);
    }
  }
  fprintf(outfile, "\n");

  for (unsigned long i=0; i<mainVector->nComponents(); i++) {

    const coStruct* thecoStruct;

	  if (getcoStruct(i, thecoStruct)) {
		  
		  for (list<unsigned long>::const_iterator pULong = thecoStruct->operatorNumbersList.begin(); pULong != thecoStruct->operatorNumbersList.end(); pULong++) {
			  
				unsigned long previousComponents = 0;
				for(unsigned long componentCounter = 0; componentCounter<i; componentCounter++) {
					  previousComponents+=mainVector->componentLength(componentCounter);
				  }

			  if(mainVector->componentLength(i)>1) {
				  fprintf(outfile, " for(unsigned long _j = 0; _j<%li; _j++) {\n",mainVector->componentLength(i));
				  fprintf(outfile, "   _temp = _segment%li_k_operator_field[_segment%li_kop_index_pointer + %li].re\n",
						  segmentNumber, segmentNumber, *pULong);
				  fprintf(outfile, "         *_active_%s_main[_active_%s_index_pointer + _j + %li].re\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "         - _segment%li_k_operator_field[_segment%li_kop_index_pointer + %li].im\n",
						  segmentNumber, segmentNumber, *pULong);
				  fprintf(outfile, "         *_active_%s_main[_active_%s_index_pointer + _j + %li].im;\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "\n");
				  fprintf(outfile, "   _active_%s_main[_active_%s_index_pointer + _j + %li].im =\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "        _segment%li_k_operator_field[_segment%li_kop_index_pointer + %li].re\n",
						  segmentNumber, segmentNumber, *pULong);
				  fprintf(outfile, "       *_active_%s_main[_active_%s_index_pointer + _j + %li].im\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "      + _segment%li_k_operator_field[_segment%li_kop_index_pointer + %li].im\n",
						  segmentNumber, segmentNumber, *pULong);
				  fprintf(outfile, "       *_active_%s_main[_active_%s_index_pointer + _j + %li].re;\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "\n");
				  fprintf(outfile, "   _active_%s_main[_active_%s_index_pointer + _j + %li].re=_temp;\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "}\n");
			  }
			  else {
				  fprintf(outfile, " _temp = _segment%li_k_operator_field[_segment%li_kop_index_pointer + %li].re\n",
						  segmentNumber, segmentNumber, *pULong);
				  fprintf(outfile, "       *_active_%s_main[_active_%s_index_pointer + %li].re\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "       - _segment%li_k_operator_field[_segment%li_kop_index_pointer + %li].im\n",
						  segmentNumber, segmentNumber, *pULong);
				  fprintf(outfile, "       *_active_%s_main[_active_%s_index_pointer + %li].im;\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "\n");
				  fprintf(outfile, " _active_%s_main[_active_%s_index_pointer + %li].im =\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "      _segment%li_k_operator_field[_segment%li_kop_index_pointer + %li].re\n",
						  segmentNumber, segmentNumber, *pULong);
				  fprintf(outfile, "     *_active_%s_main[_active_%s_index_pointer + %li].im\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "    + _segment%li_k_operator_field[_segment%li_kop_index_pointer + %li].im\n",
						  segmentNumber, segmentNumber, *pULong);
				  fprintf(outfile, "     *_active_%s_main[_active_%s_index_pointer + %li].re;\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "\n");
				  fprintf(outfile, " _active_%s_main[_active_%s_index_pointer + %li].re=_temp;\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "\n");
			  }
		  }
	  }
  }
  fprintf(outfile, "  _segment%li_kop_index_pointer+=_segment%li_nkoperators;\n", segmentNumber, segmentNumber);
  fprintf(outfile, "  _active_%s_index_pointer+=_%s_main_ncomponents;\n", fieldName, fieldName);
  fprintf(outfile, " };\n");
  fprintf(outfile, "}\n");
  fprintf(outfile, "\n");
}

// **************************************************************************
void xmdsIntegrateIP::writeTimeDepkPropagateRoutine(
                                                    FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateIP::writeTimeDepkPropagateRoutine\n");
  }

  const unsigned long nDims = simulation()->field()->geometry()->nDims();
  const unsigned long fullSpace = simulation()->field()->geometry()->fullSpace();
  const char *const fieldName = simulation()->field()->name()->c_str();

  const xmdsVector* mainVector;

  if (!simulation()->field()->getVector("main", mainVector)) {
    throw xmdsException("Internal error in xmdsIntegrateIP::writeTimeDepkPropagateRoutine: cannot find 'main' vector");
  }

  fprintf(outfile, "// *************************\n");
  fprintf(outfile, "void _segment%li_k_propagate(\n", segmentNumber);
  fprintf(outfile, " const double& _step) {\n");
  fprintf(outfile, "\n");
  list<XMLString> threadPrivateVariables;

  for (unsigned long i=0; i<nKOperators(); i++) {
    fprintf(outfile, "complex %s;\n", KOperator(i)->c_str());
    threadPrivateVariables.push_back(*KOperator(i));
  }
  fprintf(outfile, "\n");
  fprintf(outfile, "double _temp;\n");
  fprintf(outfile, "double _temp2;\n");
  fprintf(outfile, "\n");
  threadPrivateVariables.push_back("_temp");
  threadPrivateVariables.push_back("_temp2");

  simulation()->field()->vectors2space(outfile, fullSpace, *KVectorNamesList(), "");

  simulation()->field()->openLoops(outfile, fullSpace, *KVectorNamesList(), PARALLELISE_LOOP, threadPrivateVariables);

  char indent[64];
  for (unsigned long i=0; i<nDims; i++) {
    indent[i]=0x09;
  }
  indent[nDims]=0;

  fprintf(outfile, "\n");
  fprintf(outfile, "// ********** Code from k_operators *************\n");
  fprintf(outfile, "%s\n", KOperatorsCode()->c_str());
  fprintf(outfile, "// **********************************************\n");
  fprintf(outfile, "\n");

  for (unsigned long i=0; i<nKOperators(); i++) {
    fprintf(outfile, "%s_temp2 = exp(%s.re*_step);\n", indent, KOperator(i)->c_str());
    fprintf(outfile, "%s_temp = _temp2*cos(%s.im*_step);\n", indent, KOperator(i)->c_str());
    fprintf(outfile, "%s%s.im = _temp2*sin(%s.im*_step);\n", indent, KOperator(i)->c_str(), KOperator(i)->c_str());
    fprintf(outfile, "%s%s.re = _temp;\n", indent, KOperator(i)->c_str());
    fprintf(outfile, "\n");
  }
  for (unsigned long i=0; i<mainVector->nComponents(); i++) {

    const coStruct* thecoStruct;

    if (getcoStruct(i, thecoStruct)) {

		for (list<unsigned long>::const_iterator pULong = thecoStruct->operatorNumbersList.begin(); pULong != thecoStruct->operatorNumbersList.end(); pULong++) {
			
			unsigned long previousComponents = 0;
			for(unsigned long componentCounter = 0; componentCounter<i; componentCounter++) {
				previousComponents+=mainVector->componentLength(componentCounter);
			}
			if(mainVector->componentLength(i)>1) {
				fprintf(outfile, " for(unsigned long _j = 0; _j<%li; _j++) {\n",mainVector->componentLength(i));
				fprintf(outfile, "%s_temp =  %s.re*_active_%s_main[_%s_main_index_pointer + _j + %li].re\n", indent, KOperator(*pULong)->c_str(), fieldName, fieldName, previousComponents);
				fprintf(outfile, "%s       - %s.im*_active_%s_main[_%s_main_index_pointer + _j + %li].im;\n", indent, KOperator(*pULong)->c_str(), fieldName, fieldName, previousComponents);
				fprintf(outfile, "\n");
				fprintf(outfile, "%s_active_%s_main[_%s_main_index_pointer + _j + %li].im =  %s.re*_active_%s_main[_%s_main_index_pointer + _j + %li].im\n",
						indent, fieldName, fieldName, previousComponents, KOperator(*pULong)->c_str(), fieldName, fieldName, previousComponents);
				fprintf(outfile, "%s                                             + %s.im*_active_%s_main[_%s_main_index_pointer + _j + %li].re;\n",
						indent, KOperator(*pULong)->c_str(), fieldName, fieldName, previousComponents);
				fprintf(outfile, "\n");
				fprintf(outfile, "%s_active_%s_main[_%s_main_index_pointer + _j + %li].re=_temp;\n", indent, fieldName, fieldName, previousComponents);
				fprintf(outfile, "\n");
				fprintf(outfile, "%s  }\n",indent);
			}
			else {
				fprintf(outfile, "%s_temp =  %s.re*_active_%s_main[_%s_main_index_pointer + %li].re\n", indent, KOperator(*pULong)->c_str(), fieldName, fieldName, previousComponents);
				fprintf(outfile, "%s       - %s.im*_active_%s_main[_%s_main_index_pointer + %li].im;\n", indent, KOperator(*pULong)->c_str(), fieldName, fieldName, previousComponents);
				fprintf(outfile, "\n");
				fprintf(outfile, "%s_active_%s_main[_%s_main_index_pointer + %li].im =  %s.re*_active_%s_main[_%s_main_index_pointer + %li].im\n",
						indent, fieldName, fieldName, previousComponents, KOperator(*pULong)->c_str(), fieldName, fieldName, previousComponents);
				fprintf(outfile, "%s                                             + %s.im*_active_%s_main[_%s_main_index_pointer + %li].re;\n",
						indent, KOperator(*pULong)->c_str(), fieldName, fieldName, previousComponents);
				fprintf(outfile, "\n");
				fprintf(outfile, "%s_active_%s_main[_%s_main_index_pointer + %li].re=_temp;\n", indent, fieldName, fieldName, previousComponents);
				fprintf(outfile, "\n");
			}
		}
    }
  }

  simulation()->field()->closeLoops(outfile, fullSpace, *KVectorNamesList());

  fprintf(outfile, "}\n");
  fprintf(outfile, "\n");
}

// **************************************************************************
void xmdsIntegrateIP::writeARK45CalculatekOperatorFieldRoutine(
                                                               FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateARK45IP::writeCalculatekOperatorFieldRoutine\n");
  }

  const unsigned long nDims = simulation()->field()->geometry()->nDims();
  const unsigned long fullSpace = simulation()->field()->geometry()->fullSpace();

  unsigned long i;

  fprintf(outfile, "// *************************\n");
  fprintf(outfile, "void _segment%li_calculate_k_operator_field(double _step) {\n", segmentNumber);
  fprintf(outfile, "\n");
  for (i=0; i<nKOperators(); i++) {
    fprintf(outfile, "complex %s;\n", KOperator(i)->c_str());
  }
  fprintf(outfile, "\n");
  fprintf(outfile, "double _factor1=1.0/5;   // a_raw[2]\n");
  fprintf(outfile, "double _factor2=1.0/10;  // a_raw[3]-a_raw[2]\n");
  fprintf(outfile, "double _factor3=2.0/5;   // a_raw[4]-a_raw[2]\n");
  fprintf(outfile, "double _factor4=4.0/5;   // a_raw[5]-a_raw[2]\n");
  fprintf(outfile, "double _factor5=27.0/40; // a_raw[6]-a_raw[2]\n");

  list<XMLString> vectorNames(*KVectorNamesList());

  // These are used so that openLoops sets up the k_operator index correctly for OpenMP
  vectorNames.push_back(XMLString("k_operator"));
  fprintf(outfile, "unsigned long _%s_k_operator_ncomponents = _segment%li_nkoperators;\n", simulation()->field()->name()->c_str(), segmentNumber);
  list<XMLString> threadPrivateVariables;
  for (i=0; i<nKOperators(); i++)
    threadPrivateVariables.push_back(*KOperator(i));

  simulation()->field()->vectors2space(outfile, fullSpace, *KVectorNamesList(), "");

  //  fprintf(outfile, "unsigned long _k_operator_index_pointer=0;\n");

  simulation()->field()->openLoops(outfile, fullSpace, vectorNames, PARALLELISE_LOOP, threadPrivateVariables);

  char indent[64];
  for (i=0; i<nDims; i++) {
    indent[i]=0x09;
  }
  indent[nDims]=0;

  fprintf(outfile, "\n");
  fprintf(outfile, "// ********** Code from k_operators *************\n");
  fprintf(outfile, "%s\n", KOperatorsCode()->c_str());
  fprintf(outfile, "// **********************************************\n");
  fprintf(outfile, "\n");

  for (int n=1; n<6; n++){
    for (i=0; i<nKOperators(); i++) {
      fprintf(outfile, "%s_segment%li_k_operator_field_%i[_%s_k_operator_index_pointer + %li].re = exp(%s.re*_factor%i*_step)*cos(%s.im*_factor%i*_step);\n",
              indent, segmentNumber, n, simulation()->field()->name()->c_str(), i, KOperator(i)->c_str(), n, KOperator(i)->c_str(), n);
      fprintf(outfile, "%s_segment%li_k_operator_field_%i[_%s_k_operator_index_pointer + %li].im = exp(%s.re*_factor%i*_step)*sin(%s.im*_factor%i*_step);\n",
              indent, segmentNumber, n, simulation()->field()->name()->c_str(), i, KOperator(i)->c_str(), n, KOperator(i)->c_str(), n);
    }
    fprintf(outfile, "\n");
  }
  fprintf(outfile, "\n");
  //  fprintf(outfile, "%s_k_operator_index_pointer += _segment%li_nkoperators;\n", indent, segmentNumber);
  simulation()->field()->closeLoops(outfile, fullSpace, vectorNames);

  fprintf(outfile, "}\n");
  fprintf(outfile, "\n");
}


// **************************************************************************
void xmdsIntegrateIP::writeARK45kPropagateRoutine(
                                                  FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateARK45IP::writeConstkPropagateRoutine\n");
  }

  const unsigned long fullSpace = simulation()->field()->geometry()->fullSpace();
  const char *const fieldName = simulation()->field()->name()->c_str();

  list<XMLString> tempVectorNamesList;
  tempVectorNamesList.push_back("main");

  const xmdsVector* mainVector;

  if (!simulation()->field()->getVector("main", mainVector)) {
    throw xmdsException("Internal error in xmdsIntegrateARK45IP::writeConstkPropagateRoutine: cannot find 'main' vector");
  }

  fprintf(outfile, "// *************************\n");
  fprintf(outfile, "void _segment%li_k_propagate(int _exponent) {\n", segmentNumber);
  fprintf(outfile, "\n");
  fprintf(outfile, "double _temp;\n");
  fprintf(outfile, "unsigned long _segment%li_kop_index_pointer=0;\n", segmentNumber);
  fprintf(outfile, "unsigned long _active_%s_index_pointer=0;\n", fieldName);
  fprintf(outfile, "bool _firstTime = true;\n");
  fprintf(outfile, "\n");

  simulation()->field()->vectors2space(outfile, fullSpace, tempVectorNamesList, "");

  fprintf(outfile, "if (_exponent>0){\n");
  fprintf(outfile, "switch(_exponent){\n");

  for (int n=1; n<6; n++){
    fprintf(outfile, "case %i:\n", n);

    if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
      fprintf(outfile, "for (long _i0=0; _i0<total_local_size; _i0++) {\n");
    }
    else {
      if (simulation()->parameters()->useOpenMP) {
        fprintf(outfile, "#ifdef _OPENMP\n");
        fprintf(outfile, "#pragma omp parallel for firstprivate(_temp, _firstTime, _segment%li_kop_index_pointer, _active_%s_index_pointer)\n", segmentNumber, fieldName);
        fprintf(outfile, "#endif\n");
      }
      fprintf(outfile, "for (long _i0=0; _i0<_%s_size; _i0++) {\n", fieldName);
      if (simulation()->parameters()->useOpenMP) {
        fprintf(outfile, "#ifdef _OPENMP\n");
        fprintf(outfile, "if (_firstTime == true) {\n");
        fprintf(outfile, "    _firstTime = false;\n");
        fprintf(outfile, "    _segment%li_kop_index_pointer = _i0*_segment%li_nkoperators;\n", segmentNumber, segmentNumber);
        fprintf(outfile, "    _active_%s_index_pointer = _i0*_%s_main_ncomponents;\n", fieldName, fieldName);
        fprintf(outfile, "}\n");
        fprintf(outfile, "#endif\n\n");
      }
    }
    fprintf(outfile, "\n");

    for (unsigned long i=0; i<mainVector->nComponents(); i++) {
		
		const coStruct* thecoStruct;
		
		if (getcoStruct(i, thecoStruct)) {
			
			for (list<unsigned long>::const_iterator pULong = thecoStruct->operatorNumbersList.begin(); pULong != thecoStruct->operatorNumbersList.end(); pULong++) {
				
				unsigned long previousComponents = 0;
				for(unsigned long componentCounter = 0; componentCounter<i; componentCounter++) {
					previousComponents+=mainVector->componentLength(componentCounter);
				}
				
				if(mainVector->componentLength(i)>1) {
					fprintf(outfile, " for(unsigned long _j = 0; _j<%li; _j++) {\n",mainVector->componentLength(i));
					fprintf(outfile, " _temp = _segment%li_k_operator_field_%i[_segment%li_kop_index_pointer + %li].re\n",
							segmentNumber, n, segmentNumber, *pULong);
					fprintf(outfile, "       *_active_%s_main[_active_%s_index_pointer + _j + %li].re\n", fieldName, fieldName, previousComponents);
					fprintf(outfile, "       - _segment%li_k_operator_field_%i[_segment%li_kop_index_pointer + %li].im\n",
							segmentNumber, n, segmentNumber, *pULong);
					fprintf(outfile, "       *_active_%s_main[_active_%s_index_pointer + _j + %li].im;\n", fieldName, fieldName, previousComponents);
					fprintf(outfile, "\n");
					fprintf(outfile, " _active_%s_main[_active_%s_index_pointer + _j + %li].im =\n", fieldName, fieldName, previousComponents);
					fprintf(outfile, "      _segment%li_k_operator_field_%i[_segment%li_kop_index_pointer + %li].re\n",
							segmentNumber, n, segmentNumber, *pULong);
					fprintf(outfile, "     *_active_%s_main[_active_%s_index_pointer + _j + %li].im\n", fieldName, fieldName, previousComponents);
					fprintf(outfile, "    + _segment%li_k_operator_field_%i[_segment%li_kop_index_pointer + %li].im\n",
							segmentNumber, n, segmentNumber, *pULong);
					fprintf(outfile, "     *_active_%s_main[_active_%s_index_pointer + _j + %li].re;\n", fieldName, fieldName, previousComponents);
					fprintf(outfile, "\n");
					fprintf(outfile, " _active_%s_main[_active_%s_index_pointer + _j + %li].re=_temp;\n", fieldName, fieldName, previousComponents);
					fprintf(outfile, "}\n");
					fprintf(outfile, "\n");
				}
				else {
					fprintf(outfile, " _temp = _segment%li_k_operator_field_%i[_segment%li_kop_index_pointer + %li].re\n",
							segmentNumber, n, segmentNumber, *pULong);
					fprintf(outfile, "       *_active_%s_main[_active_%s_index_pointer + %li].re\n", fieldName, fieldName, previousComponents);
					fprintf(outfile, "       - _segment%li_k_operator_field_%i[_segment%li_kop_index_pointer + %li].im\n",
							segmentNumber, n, segmentNumber, *pULong);
					fprintf(outfile, "       *_active_%s_main[_active_%s_index_pointer + %li].im;\n", fieldName, fieldName, previousComponents);
					fprintf(outfile, "\n");
					fprintf(outfile, " _active_%s_main[_active_%s_index_pointer + %li].im =\n", fieldName, fieldName, previousComponents);
					fprintf(outfile, "      _segment%li_k_operator_field_%i[_segment%li_kop_index_pointer + %li].re\n",
							segmentNumber, n, segmentNumber, *pULong);
					fprintf(outfile, "     *_active_%s_main[_active_%s_index_pointer + %li].im\n", fieldName, fieldName, previousComponents);
					fprintf(outfile, "    + _segment%li_k_operator_field_%i[_segment%li_kop_index_pointer + %li].im\n",
							segmentNumber, n, segmentNumber, *pULong);
					fprintf(outfile, "     *_active_%s_main[_active_%s_index_pointer + %li].re;\n", fieldName, fieldName, previousComponents);
					fprintf(outfile, "\n");
					fprintf(outfile, " _active_%s_main[_active_%s_index_pointer + %li].re=_temp;\n", fieldName, fieldName, previousComponents);
					fprintf(outfile, "\n");
				}
			}
		}
    }
    fprintf(outfile, "  _segment%li_kop_index_pointer+=_segment%li_nkoperators;\n", segmentNumber, segmentNumber);
    fprintf(outfile, "  _active_%s_index_pointer+=_%s_main_ncomponents;\n", fieldName, fieldName);
    fprintf(outfile, " };\n");
    fprintf(outfile, " break;\n");
  }
  fprintf(outfile,
      "  }\n"
      "}\n"
      "else {\n"
      "_exponent=abs(_exponent);\n"
      "switch(_exponent){\n");

  for (int n=1; n<6; n++){
    fprintf(outfile, "case %i:\n", n);

    if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
      fprintf(outfile, "for (long _i0=0; _i0<total_local_size; _i0++) {\n");
    }
    else {
      if (simulation()->parameters()->useOpenMP) {
        fprintf(outfile, "#ifdef _OPENMP\n");
        fprintf(outfile, "#pragma omp parallel for firstprivate(_temp, _firstTime, _segment%li_kop_index_pointer, _active_%s_index_pointer)\n", segmentNumber, fieldName);
        fprintf(outfile, "#endif\n");
      }
      fprintf(outfile, "for (long _i0=0; _i0<_%s_size; _i0++) {\n", fieldName);
      if (simulation()->parameters()->useOpenMP) {
        fprintf(outfile, "#ifdef _OPENMP\n");
        fprintf(outfile, "if (_firstTime == true) {\n");
        fprintf(outfile, "    _firstTime = false;\n");
        fprintf(outfile, "    _segment%li_kop_index_pointer = _i0*_segment%li_nkoperators;\n", segmentNumber, segmentNumber);
        fprintf(outfile, "    _active_%s_index_pointer = _i0*_%s_main_ncomponents;\n", fieldName, fieldName);
        fprintf(outfile, "}\n");
        fprintf(outfile, "#endif\n\n");
      }
    }
    fprintf(outfile, "\n");

    for (unsigned long i=0; i<mainVector->nComponents(); i++) {

      const coStruct* thecoStruct;

      if (getcoStruct(i, thecoStruct)) {

		  for (list<unsigned long>::const_iterator pULong = thecoStruct->operatorNumbersList.begin(); pULong != thecoStruct->operatorNumbersList.end(); pULong++) {
			  
			  unsigned long previousComponents = 0;
			  for(unsigned long componentCounter = 0; componentCounter<i; componentCounter++) {
				  previousComponents+=mainVector->componentLength(componentCounter);
			  }
			  
			  if(mainVector->componentLength(i)>1) {
				  fprintf(outfile, " for(unsigned long _j = 0; _j<%li; _j++) {\n",mainVector->componentLength(i));
				  fprintf(outfile, "    _active_%s_main[_active_%s_index_pointer + _j + %li]/=_segment%li_k_operator_field_%i[_segment%li_kop_index_pointer + %li];\n", fieldName, fieldName, previousComponents, segmentNumber, n, segmentNumber, *pULong);
				  fprintf(outfile, "}\n");
				  fprintf(outfile, "\n");
			  }
			  else {
				  fprintf(outfile, " _active_%s_main[_active_%s_index_pointer + %li]/=_segment%li_k_operator_field_%i[_segment%li_kop_index_pointer + %li];\n", fieldName, fieldName, previousComponents, segmentNumber, n, segmentNumber, *pULong);
				  fprintf(outfile, "\n");
			  }
		  }
      }
    }
    fprintf(outfile, "  _segment%li_kop_index_pointer+=_segment%li_nkoperators;\n", segmentNumber, segmentNumber);
    fprintf(outfile, "  _active_%s_index_pointer+=_%s_main_ncomponents;\n", fieldName, fieldName);
    fprintf(outfile, " };\n");
    fprintf(outfile, " break;\n");
  }
  fprintf(outfile, "}\n");
  fprintf(outfile, "}\n");
  fprintf(outfile, "}\n");
  fprintf(outfile, "\n");
}

// **************************************************************************
void xmdsIntegrateIP::writeARK45TimeDepkCalculatekOperatorFieldRoutine(
                                                                       FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateARK45IP::writeARK45TimeDepkCalculatekOperatorFieldRoutine\n");
  }

  const unsigned long nDims = simulation()->field()->geometry()->nDims();
  const char *const propDim = simulation()->parameters()->propDimName.c_str();
  const unsigned long fullSpace = simulation()->field()->geometry()->fullSpace();

  unsigned long i;

  fprintf(outfile, "// *************************\n");
  fprintf(outfile, "void _segment%li_calculate_k_operator_field(double _step) {\n", segmentNumber);
  fprintf(outfile, "\n");
  for (i=0; i<nKOperators(); i++) {
    fprintf(outfile, "complex %s;\n", KOperator(i)->c_str());
  }
  fprintf(outfile, "\n");
  fprintf(outfile, "double factor1=1.0/5;   // a_raw[2]\n");
  fprintf(outfile, "double factor2=1.0/10;  // a_raw[3]-a_raw[2]\n");
  fprintf(outfile, "double factor3=2.0/5;   // a_raw[4]-a_raw[2]\n");
  fprintf(outfile, "double factor4=4.0/5;   // a_raw[5]-a_raw[2]\n");
  fprintf(outfile, "double factor5=27.0/40; // a_raw[6]-a_raw[2]\n");
  fprintf(outfile, "double _store_time=t;\n");

  fprintf(outfile, "\n");
  fprintf(outfile, "double _store_%s=%s; \n", propDim, propDim);
  fprintf(outfile, "double _d%s1=0.0;   \n", propDim);
  fprintf(outfile, "double _d%s2=3.0/10*_step;   \n", propDim);
  fprintf(outfile, "double _d%s3=3.0/10*_step;  \n", propDim);
  fprintf(outfile, "double _d%s4=2.0/5*_step;   \n", propDim);
  fprintf(outfile, "double _d%s5=-1.0/8*_step;   \n", propDim);
  fprintf(outfile, "\n");

  simulation()->field()->vectors2space(outfile, fullSpace, *KVectorNamesList(), "");

  fprintf(outfile, "unsigned long _k_operator_index_pointer;\n");

  for (int n=1; n<6; n++){
    fprintf(outfile, "_k_operator_index_pointer=0;\n");
    fprintf(outfile, "%s+=_d%s%i;\n", propDim, propDim, n);
    fprintf(outfile, "\n");
    fprintf(outfile, "{\n"); //make counters in loops local

    simulation()->field()->openLoops(outfile, fullSpace, *KVectorNamesList());

    char indent[64];
    for (i=0; i<nDims; i++) {
      indent[i]=0x09;
    }
    indent[nDims]=0;

    fprintf(outfile, "\n");
    fprintf(outfile, "// ********** Code from k_operators *************\n");
    fprintf(outfile, "%s\n", KOperatorsCode()->c_str());
    fprintf(outfile, "// **********************************************\n");
    fprintf(outfile, "\n");


    for (i=0; i<nKOperators(); i++) {
      fprintf(outfile, "%s_segment%li_k_operator_field_%i[_k_operator_index_pointer + %li].re = exp(%s.re*factor%i*_step)*cos(%s.im*factor%i*_step);\n",
              indent, segmentNumber, n, i, KOperator(i)->c_str(), n, KOperator(i)->c_str(), n);
      fprintf(outfile, "%s_segment%li_k_operator_field_%i[_k_operator_index_pointer + %li].im = exp(%s.re*factor%i*_step)*sin(%s.im*factor%i*_step);\n",
              indent, segmentNumber, n, i, KOperator(i)->c_str(), n, KOperator(i)->c_str(), n);
    }

    fprintf(outfile, "\n");
    fprintf(outfile, "%s_k_operator_index_pointer += _segment%li_nkoperators;\n", indent, segmentNumber);
    simulation()->field()->closeLoops(outfile, fullSpace, *KVectorNamesList());
    fprintf(outfile, "}\n"); // end-make counters in loops local

    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s=_store_%s;\n", propDim, propDim);
  fprintf(outfile, "}\n");
  fprintf(outfile, "\n");
}

// **************************************************************************
void xmdsIntegrateIP::writeRK9CalculatekOperatorFieldRoutine(
                                                             FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateRK9IP::writeCalculatekOperatorFieldRoutine\n");
  }

  const unsigned long nDims = simulation()->field()->geometry()->nDims();
  const unsigned long fullSpace = simulation()->field()->geometry()->fullSpace();

  unsigned long i;

  fprintf(outfile, "// *************************\n");
  fprintf(outfile, "void _segment%li_calculate_k_operator_field(double _step) {\n", segmentNumber);
  fprintf(outfile, "\n");
  for (i=0; i<nKOperators(); i++) {
    fprintf(outfile, "complex %s;\n", KOperator(i)->c_str());
  }
  fprintf(outfile, "\n");
  fprintf(outfile, "double factor1=-1.000000000000000;  // a_raw[15]-a_raw[1]\n");
  fprintf(outfile, "double factor2=-0.978260869565217;  // a_raw[15]-a_raw[2]\n");
  fprintf(outfile, "double factor3=-0.903704189521999;  // a_raw[15]-a_raw[3]\n");
  fprintf(outfile, "double factor4=-0.855556284282999;  // a_raw[15]-a_raw[4]\n");
  fprintf(outfile, "double factor5=-0.477941176470588;  // a_raw[15]-a_raw[5]\n");
  fprintf(outfile, "double factor6=-0.771575563871365;  // a_raw[15]-a_raw[6]\n");
  fprintf(outfile, "double factor7=-0.456396464100663;  // a_raw[15]-a_raw[7]\n");
  fprintf(outfile, "double factor8=-0.356643356643357;  // a_raw[15]-a_raw[8]\n");
  fprintf(outfile, "double factor9=-0.517482517482518;  // a_raw[15]-a_raw[9]\n");
  fprintf(outfile, "double factor10=-0.931818181818182; // a_raw[15]-a_raw[10]\n");
  fprintf(outfile, "double factor11=-0.749391727493917; // a_raw[15]-a_raw[11]\n");
  fprintf(outfile, "double factor12=-0.332632840343994; // a_raw[15]-a_raw[12]\n");
  fprintf(outfile, "double factor13=-0.144927536231884; // a_raw[15]-a_raw[13]\n");
  fprintf(outfile, "double factor14=-0.102040816326531; // a_raw[15]-a_raw[14]\n");


  list<XMLString> vectorNames(*KVectorNamesList());

  // These are used so that openLoops sets up the k_operator index correctly for OpenMP
  vectorNames.push_back(XMLString("k_operator"));
  fprintf(outfile, "unsigned long _%s_k_operator_ncomponents = _segment%li_nkoperators;\n", simulation()->field()->name()->c_str(), segmentNumber);
  list<XMLString> threadPrivateVariables;
  for (i=0; i<nKOperators(); i++)
    threadPrivateVariables.push_back(*KOperator(i));

  simulation()->field()->vectors2space(outfile, fullSpace, *KVectorNamesList(), "");

  //  fprintf(outfile, "unsigned long _k_operator_index_pointer=0;\n");

  simulation()->field()->openLoops(outfile, fullSpace, vectorNames, PARALLELISE_LOOP, threadPrivateVariables);

  char indent[64];
  for (i=0; i<nDims; i++) {
    indent[i]=0x09;
  }
  indent[nDims]=0;

  fprintf(outfile, "\n");
  fprintf(outfile, "// ********** Code from k_operators *************\n");
  fprintf(outfile, "%s\n", KOperatorsCode()->c_str());
  fprintf(outfile, "// **********************************************\n");
  fprintf(outfile, "\n");

  for (int n=1; n<15; n++){
    for (i=0; i<nKOperators(); i++) {
      fprintf(outfile, "%s_segment%li_k_operator_field_%i[_%s_k_operator_index_pointer + %li].re = exp(%s.re*factor%i*_step)*cos(%s.im*factor%i*_step);\n",
              indent, segmentNumber, n, simulation()->field()->name()->c_str(), i, KOperator(i)->c_str(), n, KOperator(i)->c_str(), n);
      fprintf(outfile, "%s_segment%li_k_operator_field_%i[_%s_k_operator_index_pointer + %li].im = exp(%s.re*factor%i*_step)*sin(%s.im*factor%i*_step);\n",
              indent, segmentNumber, n, simulation()->field()->name()->c_str(), i, KOperator(i)->c_str(), n, KOperator(i)->c_str(), n);
    }
    fprintf(outfile, "\n");
  }
  fprintf(outfile, "\n");
  //  fprintf(outfile, "%s_k_operator_index_pointer += _segment%li_nkoperators;\n", indent, segmentNumber);
  simulation()->field()->closeLoops(outfile, fullSpace, vectorNames);

  fprintf(outfile, "}\n");
  fprintf(outfile, "\n");
}


// **************************************************************************
void xmdsIntegrateIP::writeRK9kPropagateRoutine(
                                                FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateRK9IP::writeConstkPropagateRoutine\n");
  }

  const unsigned long fullSpace = simulation()->field()->geometry()->fullSpace();
  const char *const fieldName = simulation()->field()->name()->c_str();

  list<XMLString> tempVectorNamesList;
  tempVectorNamesList.push_back("main");

  const xmdsVector* mainVector;

  if (!simulation()->field()->getVector("main", mainVector)) {
    throw xmdsException("Internal error in xmdsIntegrateRK9IP::writeConstkPropagateRoutine: cannot find 'main' vector");
  }

  fprintf(outfile, "// *************************\n");
  fprintf(outfile, "void _segment%li_k_propagate(int exponent) {\n", segmentNumber);
  fprintf(outfile, "\n");
  fprintf(outfile, "double _temp;\n");
  fprintf(outfile, "unsigned long _segment%li_kop_index_pointer=0;\n", segmentNumber);
  fprintf(outfile, "unsigned long _active_%s_index_pointer=0;\n", fieldName);
  fprintf(outfile, "bool _firstTime = true;\n");
  fprintf(outfile, "\n");

  simulation()->field()->vectors2space(outfile, fullSpace, tempVectorNamesList, "");

  fprintf(outfile, "if (exponent>0){\n");
  fprintf(outfile, "switch(exponent){\n");

  for (int n=1; n<15; n++){
    fprintf(outfile, "case %i:\n", n);

    if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
      fprintf(outfile, "for (long _i0=0; _i0<total_local_size; _i0++) {\n");
    }
    else {
      if (simulation()->parameters()->useOpenMP) {
        fprintf(outfile, "#ifdef _OPENMP\n");
        fprintf(outfile, "#pragma omp parallel for firstprivate(_temp, _firstTime, _segment%li_kop_index_pointer, _active_%s_index_pointer)\n", segmentNumber, fieldName);
        fprintf(outfile, "#endif\n");
      }
      fprintf(outfile, "for (long _i0=0; _i0<_%s_size; _i0++) {\n", fieldName);
      if (simulation()->parameters()->useOpenMP) {
        fprintf(outfile, "#ifdef _OPENMP\n");
        fprintf(outfile, "if (_firstTime == true) {\n");
        fprintf(outfile, "    _firstTime = false;\n");
        fprintf(outfile, "    _segment%li_kop_index_pointer = _i0*_segment%li_nkoperators;\n", segmentNumber, segmentNumber);
        fprintf(outfile, "    _active_%s_index_pointer = _i0*_%s_main_ncomponents;\n", fieldName, fieldName);
        fprintf(outfile, "}\n");
        fprintf(outfile, "#endif\n\n");
      }
    }
    fprintf(outfile, "\n");

    for (unsigned long i=0; i<mainVector->nComponents(); i++) {

      const coStruct* thecoStruct;

      if (getcoStruct(i, thecoStruct)) {

		  for (list<unsigned long>::const_iterator pULong = thecoStruct->operatorNumbersList.begin(); pULong != thecoStruct->operatorNumbersList.end(); pULong++) {
			  
			  unsigned long previousComponents = 0;
			  for(unsigned long componentCounter = 0; componentCounter<i; componentCounter++) {
				  previousComponents+=mainVector->componentLength(componentCounter);
			  }
			  
			  if(mainVector->componentLength(i)>1) {
				  fprintf(outfile, " for(unsigned long _j = 0; _j<%li; _j++) {\n",mainVector->componentLength(i));
				  fprintf(outfile, " _temp = _segment%li_k_operator_field_%i[_segment%li_kop_index_pointer + %li].re\n",
						  segmentNumber, n, segmentNumber, *pULong);
				  fprintf(outfile, "       *_active_%s_main[_active_%s_index_pointer + _j + %li].re\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "       - _segment%li_k_operator_field_%i[_segment%li_kop_index_pointer + %li].im\n",
						  segmentNumber, n, segmentNumber, *pULong);
				  fprintf(outfile, "       *_active_%s_main[_active_%s_index_pointer + _j + %li].im;\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "\n");
				  fprintf(outfile, " _active_%s_main[_active_%s_index_pointer + _j + %li].im =\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "      _segment%li_k_operator_field_%i[_segment%li_kop_index_pointer + %li].re\n",
						  segmentNumber, n, segmentNumber, *pULong);
				  fprintf(outfile, "     *_active_%s_main[_active_%s_index_pointer + _j + %li].im\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "    + _segment%li_k_operator_field_%i[_segment%li_kop_index_pointer + %li].im\n",
						  segmentNumber, n, segmentNumber, *pULong);
				  fprintf(outfile, "     *_active_%s_main[_active_%s_index_pointer + _j + %li].re;\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "\n");
				  fprintf(outfile, " _active_%s_main[_active_%s_index_pointer + _j + %li].re=_temp;\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "}\n");
				  fprintf(outfile, "\n");
			  }
			  else {
				  fprintf(outfile, " _temp = _segment%li_k_operator_field_%i[_segment%li_kop_index_pointer + %li].re\n",
						  segmentNumber, n, segmentNumber, *pULong);
				  fprintf(outfile, "       *_active_%s_main[_active_%s_index_pointer + %li].re\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "       - _segment%li_k_operator_field_%i[_segment%li_kop_index_pointer + %li].im\n",
						  segmentNumber, n, segmentNumber, *pULong);
				  fprintf(outfile, "       *_active_%s_main[_active_%s_index_pointer + %li].im;\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "\n");
				  fprintf(outfile, " _active_%s_main[_active_%s_index_pointer + %li].im =\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "      _segment%li_k_operator_field_%i[_segment%li_kop_index_pointer + %li].re\n",
						  segmentNumber, n, segmentNumber, *pULong);
				  fprintf(outfile, "     *_active_%s_main[_active_%s_index_pointer + %li].im\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "    + _segment%li_k_operator_field_%i[_segment%li_kop_index_pointer + %li].im\n",
						  segmentNumber, n, segmentNumber, *pULong);
				  fprintf(outfile, "     *_active_%s_main[_active_%s_index_pointer + %li].re;\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "\n");
				  fprintf(outfile, " _active_%s_main[_active_%s_index_pointer + %li].re=_temp;\n", fieldName, fieldName, previousComponents);
				  fprintf(outfile, "\n");
			  }
		  }
      }
    }
    fprintf(outfile, "  _segment%li_kop_index_pointer+=_segment%li_nkoperators;\n", segmentNumber, segmentNumber);
    fprintf(outfile, "  _active_%s_index_pointer+=_%s_main_ncomponents;\n", fieldName, fieldName);
    fprintf(outfile, " };\n");
    fprintf(outfile, " break;\n");
  }
  fprintf(outfile,
      "  }\n"
      "}\n"
      "else {\n"
      "exponent=abs(exponent);\n"
      "switch(exponent){\n");

  for (int n=1; n<15; n++){
    fprintf(outfile, "case %i:\n", n);

    if (simulation()->parameters()->usempi&!simulation()->parameters()->stochastic) {
      fprintf(outfile, "for (long _i0=0; _i0<total_local_size; _i0++) {\n");
    }
    else {
      if (simulation()->parameters()->useOpenMP) {
        fprintf(outfile, "#ifdef _OPENMP\n");
        fprintf(outfile, "#pragma omp parallel for firstprivate(_temp, _firstTime, _segment%li_kop_index_pointer, _active_%s_index_pointer)\n", segmentNumber, fieldName);
        fprintf(outfile, "#endif\n");
      }
      fprintf(outfile, "for (long _i0=0; _i0<_%s_size; _i0++) {\n", fieldName);
      if (simulation()->parameters()->useOpenMP) {
        fprintf(outfile, "#ifdef _OPENMP\n");
        fprintf(outfile, "if (_firstTime == true) {\n");
        fprintf(outfile, "    _firstTime = false;\n");
        fprintf(outfile, "    _segment%li_kop_index_pointer = _i0*_segment%li_nkoperators;\n", segmentNumber, segmentNumber);
        fprintf(outfile, "    _active_%s_index_pointer = _i0*_%s_main_ncomponents;\n", fieldName, fieldName);
        fprintf(outfile, "}\n");
        fprintf(outfile, "#endif\n\n");
      }
    }
    fprintf(outfile, "\n");

    for (unsigned long i=0; i<mainVector->nComponents(); i++) {

      const coStruct* thecoStruct;

      if (getcoStruct(i, thecoStruct)) {

		  for (list<unsigned long>::const_iterator pULong = thecoStruct->operatorNumbersList.begin(); pULong != thecoStruct->operatorNumbersList.end(); pULong++) {
			  
			  unsigned long previousComponents = 0;
			  for(unsigned long componentCounter = 0; componentCounter<i; componentCounter++) {
				  previousComponents+=mainVector->componentLength(componentCounter);
			  }
			  
			  if(mainVector->componentLength(i)>1) {
				  fprintf(outfile, " for(unsigned long _j = 0; _j<%li; _j++) {\n",mainVector->componentLength(i));
				  fprintf(outfile, "   _active_%s_main[_active_%s_index_pointer + _j + %li]/=_segment%li_k_operator_field_%i[_segment%li_kop_index_pointer + %li];\n", fieldName, fieldName, previousComponents, segmentNumber, n, segmentNumber, *pULong);
				  fprintf(outfile, "}\n");
				  fprintf(outfile, "\n");
			  }
			  else {
				  fprintf(outfile, " _active_%s_main[_active_%s_index_pointer + %li]/=_segment%li_k_operator_field_%i[_segment%li_kop_index_pointer + %li];\n", fieldName, fieldName, previousComponents, segmentNumber, n, segmentNumber, *pULong);
				  fprintf(outfile, "\n");
			  }
		  }
      }
    }
    fprintf(outfile, "  _segment%li_kop_index_pointer+=_segment%li_nkoperators;\n", segmentNumber, segmentNumber);
    fprintf(outfile, "  _active_%s_index_pointer+=_%s_main_ncomponents;\n", fieldName, fieldName);
    fprintf(outfile, " };\n");
    fprintf(outfile, " break;\n");
  }
  fprintf(outfile, "}\n");
  fprintf(outfile, "}\n");
  fprintf(outfile, "}\n");
  fprintf(outfile, "\n");
}

// **************************************************************************
void xmdsIntegrateIP::writeRK9TimeDepkCalculatekOperatorFieldRoutine(
                                                                     FILE *const outfile) const {
  if (debugFlag) {
    printf("xmdsIntegrateRK9IP::writeARK45TimeDepkCalculatekOperatorFieldRoutine\n");
  }

  const unsigned long nDims = simulation()->field()->geometry()->nDims();
  const char *const propDim = simulation()->parameters()->propDimName.c_str();
  const unsigned long fullSpace = simulation()->field()->geometry()->fullSpace();

  unsigned long i;

  fprintf(outfile, "// *************************\n");
  fprintf(outfile, "void _segment%li_calculate_k_operator_field(double _step) {\n", segmentNumber);
  fprintf(outfile, "\n");
  for (i=0; i<nKOperators(); i++) {
    fprintf(outfile, "complex %s;\n", KOperator(i)->c_str());
  }
  fprintf(outfile, "\n");
  fprintf(outfile, "double factor1=-1.000000000000000;  // a_raw[15]-a_raw[1]\n");
  fprintf(outfile, "double factor2=-0.978260869565217;  // a_raw[15]-a_raw[2]\n");
  fprintf(outfile, "double factor3=-0.903704189521999;  // a_raw[15]-a_raw[3]\n");
  fprintf(outfile, "double factor4=-0.855556284282999;  // a_raw[15]-a_raw[4]\n");
  fprintf(outfile, "double factor5=-0.477941176470588;  // a_raw[15]-a_raw[5]\n");
  fprintf(outfile, "double factor6=-0.771575563871365;  // a_raw[15]-a_raw[6]\n");
  fprintf(outfile, "double factor7=-0.456396464100663;  // a_raw[15]-a_raw[7]\n");
  fprintf(outfile, "double factor8=-0.356643356643357;  // a_raw[15]-a_raw[8]\n");
  fprintf(outfile, "double factor9=-0.517482517482518;  // a_raw[15]-a_raw[9]\n");
  fprintf(outfile, "double factor10=-0.931818181818182; // a_raw[15]-a_raw[10]\n");
  fprintf(outfile, "double factor11=-0.749391727493917; // a_raw[15]-a_raw[11]\n");
  fprintf(outfile, "double factor12=-0.332632840343994; // a_raw[15]-a_raw[12]\n");
  fprintf(outfile, "double factor13=-0.144927536231884; // a_raw[15]-a_raw[13]\n");
  fprintf(outfile, "double factor14=-0.102040816326531; // a_raw[15]-a_raw[14]\n");
  fprintf(outfile, "double _store_time=t;\n");

  fprintf(outfile, "\n");
  fprintf(outfile, "double _store_%s=%s; \n", propDim, propDim);
  fprintf(outfile, "double _d%s1=1.000000000000000*_step;  \n", propDim);
  fprintf(outfile, "double _d%s2=0.978260869565217*_step;  \n", propDim);
  fprintf(outfile, "double _d%s3=0.903704189521999*_step;  \n", propDim);
  fprintf(outfile, "double _d%s4=0.855556284282999*_step;  \n", propDim);
  fprintf(outfile, "double _d%s5=0.477941176470588*_step;  \n", propDim);
  fprintf(outfile, "double _d%s6=0.771575563871365*_step;  \n", propDim);
  fprintf(outfile, "double _d%s7=0.456396464100663*_step;  \n", propDim);
  fprintf(outfile, "double _d%s8=0.356643356643357*_step;  \n", propDim);
  fprintf(outfile, "double _d%s9=0.517482517482518*_step;  \n", propDim);
  fprintf(outfile, "double _d%s10=0.931818181818182*_step; \n", propDim);
  fprintf(outfile, "double _d%s11=0.749391727493917*_step; \n", propDim);
  fprintf(outfile, "double _d%s12=0.332632840343994*_step; \n", propDim);
  fprintf(outfile, "double _d%s13=0.144927536231884*_step; \n", propDim);
  fprintf(outfile, "double _d%s14=0.102040816326531*_step; \n", propDim);
  fprintf(outfile, "\n");

  simulation()->field()->vectors2space(outfile, fullSpace, *KVectorNamesList(), "");

  fprintf(outfile, "unsigned long _k_operator_index_pointer;\n");

  for (int n=1; n<15; n++){
    fprintf(outfile, "_k_operator_index_pointer=0;\n");
    fprintf(outfile, "%s+=_d%s%i;\n", propDim, propDim, n);
    fprintf(outfile, "\n");
    fprintf(outfile, "{\n"); //make counters in loops local

    simulation()->field()->openLoops(outfile, fullSpace, *KVectorNamesList());

    char indent[64];
    for (i=0; i<nDims; i++) {
      indent[i]=0x09;
    }
    indent[nDims]=0;

    fprintf(outfile, "\n");
    fprintf(outfile, "// ********** Code from k_operators *************\n");
    fprintf(outfile, "%s\n", KOperatorsCode()->c_str());
    fprintf(outfile, "// **********************************************\n");
    fprintf(outfile, "\n");


    for (i=0; i<nKOperators(); i++) {
      fprintf(outfile, "%s_segment%li_k_operator_field_%i[_k_operator_index_pointer + %li].re = exp(%s.re*factor%i*_step)*cos(%s.im*factor%i*_step);\n",
              indent, segmentNumber, n, i, KOperator(i)->c_str(), n, KOperator(i)->c_str(), n);
      fprintf(outfile, "%s_segment%li_k_operator_field_%i[_k_operator_index_pointer + %li].im = exp(%s.re*factor%i*_step)*sin(%s.im*factor%i*_step);\n",
              indent, segmentNumber, n, i, KOperator(i)->c_str(), n, KOperator(i)->c_str(), n);
    }

    fprintf(outfile, "\n");
    fprintf(outfile, "%s_k_operator_index_pointer += _segment%li_nkoperators;\n", indent, segmentNumber);
    simulation()->field()->closeLoops(outfile, fullSpace, *KVectorNamesList());
    fprintf(outfile, "}\n"); // end-make counters in loops local

    fprintf(outfile, "\n");
  }

  fprintf(outfile, "%s=_store_%s;\n", propDim, propDim);
  fprintf(outfile, "}\n");
  fprintf(outfile, "\n");
}

/*
 * Local variables:
 * c-indentation-style: bsd
 * c-basic-offset: 2
 * indent-tabs-mode: nil
 * End:
 *
 * vim: tabstop=2 expandtab shiftwidth=2:
 */
