/*

    This file is part of the Maude 3 interpreter.

    Copyright 2020 SRI International, Menlo Park, CA 94025, USA.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.G

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.

*/

//
//      Class for process manager symbols.
//
#ifndef _processManagerSymbol_hh_
#define _processManagerSymbol_hh_
#include <sys/types.h>
#include <map>
#include "externalObjectManagerSymbol.hh"
#include "pseudoThread.hh"
#include "socketManagerSymbol.hh"
#include "AU_Theory.hh"
#include "AU_Symbol.hh"
#include "dagRoot.hh"

class ProcessManagerSymbol
  : public ExternalObjectManagerSymbol,
    public PseudoThread
{
  NO_COPYING(ProcessManagerSymbol);

public:
  ProcessManagerSymbol(int id);

  bool attachData(const Vector<Sort*>& opDeclaration,
		  const char* purpose,
		  const Vector<const char*>& data);
  bool attachSymbol(const char* purpose, Symbol* symbol);
  void copyAttachments(Symbol* original, SymbolMap* map);
  void getDataAttachments(const Vector<Sort*>& opDeclaration,
			  Vector<const char*>& purposes,
			  Vector<Vector<const char*> >& data);
  void getSymbolAttachments(Vector<const char*>& purposes,
			    Vector<Symbol*>& symbols);
  //
  //	Overridden methods from ExternalObjectManagerSymbol.
  //
  bool handleManagerMessage(DagNode* message, ObjectSystemRewritingContext& context);
  bool handleMessage(DagNode* message, ObjectSystemRewritingContext& context);
  void cleanUp(DagNode* objectId);
  //
  //	Overridden method from PseudoThread.
  //
  void doChildExit(pid_t childPid);
  //
  //	Static function to disable/enable this class.
  //
  static void setAllowProcesses(bool flag);

private:
  enum SpecialValues
    {
     ERROR_BUFFER_SIZE = 256
    };

  struct ChildProcess
  {
    ChildProcess();
    //
    //	Sockets connected to this process.
    //
    int ioSocket;
    int errSocket;
    //
    //	If we make a request for an async notification of a child exit
    //	we need to save a pointer to the original context and the waitForExit()
    //	message so we can generate an exited() reply.
    //
    ObjectSystemRewritingContext* waitContext;
    DagRoot waitMessage;
  };

  typedef map<pid_t, ChildProcess> ProcessMap;

  bool createProcess(FreeDagNode* message, ObjectSystemRewritingContext& context);
  bool waitForExit(FreeDagNode* message, ObjectSystemRewritingContext& context);
  bool signalProcess(FreeDagNode* message, ObjectSystemRewritingContext& context);

  int checkStringList(DagNode* stringList);
  char* const* makeStringArray(DagNode*  zeroArgument,
			       DagNode* remainingArguments,
			       int nrRemainingArguments);
  bool getChildProcess(DagNode* processArg, int& processId, ChildProcess*& cpp);
  bool makeNonblockingSocketPair(int pair[2],
				 FreeDagNode* message,
				 ObjectSystemRewritingContext& context,
				 bool readOnly);
  bool makeCloseOnExitPipe(int pair[2],
			   FreeDagNode* message,
			   ObjectSystemRewritingContext& context);
  void exitedReply(pid_t processId,
		   int exitCode,
		   FreeDagNode* originalMessage,
		   ObjectSystemRewritingContext& context);
  void errorReply(const Rope& errorMessage,
		  FreeDagNode* originalMessage,
		  ObjectSystemRewritingContext& context);
    
  static int getSignalNumber(const char* signalString);
  static const char* getSignalName(int signalNumber);
  //
  //	Shared flag to disable the functionality of this class.
  //
  static bool allowProcesses;
  //
  //	Process subsystem signature (generated by macro expansion).
  //
#define MACRO(SymbolName, SymbolClass, NrArgs) \
  SymbolClass* SymbolName;
#include "processSignature.cc"
#undef MACRO

  ProcessMap childProcesses;
};

inline void
ProcessManagerSymbol::setAllowProcesses(bool flag)
{
  allowProcesses = flag;
}

inline
ProcessManagerSymbol::ChildProcess::ChildProcess()
{
  waitContext = 0;
}

#endif
