#! /bin/bash

set -e

# No need to build anything if only format is linuxdoc sgml
if [ "${BUILDDOC_FORMATS}" = "sgml" ]; then
    echo "Skip doc build. linuxdoc sgml is already present." >&2
    exit
fi

export TMPDIR=`mktemp -d ${TMPDIR:-/tmp}/ldt.XXXXXXXXXX`;

echo "-------- Building linuxdoc-tools docs ---------"
echo "Using temporary directory: $TMPDIR"

abort()
{
     sleep 1; rm -rf $TMPDIR; exit 1
}

trap 'abort' 1 2 3 6 9 15


PERL=`which perl`
TMPDATADIR=${TMPDIR}/linuxdoc-tools
mkdir -p ${TMPDATADIR}

( cd ${PKGDATADIR} && cp -r . ${TMPDATADIR} )
cp ../tex/*.sty ${TMPDATADIR}
cp ../VERSION ${TMPDATADIR}

if [ "${BUILD_ENTITY_MAP}" = "true" ]; then
    # Create a modified EntityMap.pm with entity-map location in doc
    # build temporary dir. Need to properly install entity-map there.
    mkdir $TMPDIR/Text
    make -C ../entity-map install DESTDIR="$TMPDIR"
    sed < ../entity-map/EntityMap.pm.in > $TMPDIR/Text/EntityMap.pm \
	-e 's|\@localentitymapdir\@|'${TMPDIR}'/usr/share/entity-map|g' \
	-e 's|\@entitymapdir\@|'${TMPDIR}'/usr/share/entity-map/0.1.0|g'

    # Set ${TMPDIR} first in perl load path (Will put Text dir there
    # for modified EntityMap.pm), then our perl5lib
    export PERL5LIB=${TMPDIR}:${PKGPERL5LIB}
else
    export PERL5LIB=${PKGPERL5LIB}
fi

# Set prefix for iso-entities location and make it available if needed.
if [ "${BUILD_ISO_ENTITIES}" = "true" ]; then
    # --without-installed-iso-entities: Install iso-entities in
    # "$TMPDIR/usr" and set it as iso-entities prefix.
    make -C ../iso-entities install DESTDIR="$TMPDIR"
    ISOENTITIES_PREFIX="${TMPDIR}/usr"
else
    # --with-installed-iso-entities: Use system prefix.
    ISOENTITIES_PREFIX="${PREFIX}"
fi

# Make sure our binaries are available in doc build environment
TMP_BINDIR=${TMPDIR}/bin
mkdir -p ${TMP_BINDIR}
install -m 0755 ../sgmlpre/sgmlpre ${TMP_BINDIR}
install -m 0755 ../rtf-fix/rtf2rtf ${TMP_BINDIR}
if [ -x "../sgmls-1.1/sgmlsasp" ]; then
    install -m 0755 ../sgmls-1.1/sgmlsasp ${TMP_BINDIR}
fi

export PATH=${PATH}:${TMP_BINDIR}

# Create a linuxdoc copy using our temporary locations.
sed < ../bin/linuxdoc.in > $TMPDIR/linuxdoc \
  -e 's!\@prefix\@!'${TMPDIR}/usr'!g' \
  -e 's!\@isoentities_prefix\@!'${ISOENTITIES_PREFIX}'!g' \
  -e 's!\@auxbindir\@!'${TMP_BINDIR}'!g' \
  -e 's!\@pkgdatadir\@!'${TMPDATADIR}'!g' \
  -e 's!\@perl5libdir\@!'${TMPDIR}'!g' \
  -e 's!\@GROFFMACRO\@!-ms!g' \
  -e 's!\@PERL\@!'${PERL}'!g' \
  -e 's!\@PERLWARN\@!!g'

chmod u+x $TMPDIR/linuxdoc

# Unless otherwise instructed, build docs for all formats
if [ -z "${BUILDDOC_FORMATS}" ]; then
    echo "- ++ Warning: \"\${BUILDDOC_FORMATS}\" unset. Building all doc formats:" >&2
    BUILDDOC_FORMATS="txt pdf info lyx html rtf dvi+ps"
fi

# Build actual documentation
echo "- Building documentation for formats: ${BUILDDOC_FORMATS}" >&2
for docformat in ${BUILDDOC_FORMATS}; do
    case ${docformat} in
	txt)
	    echo "- Building txt docs" >&2
	    if [ -n "`which groff`" ]; then
		$TMPDIR/linuxdoc --backend=txt --filter --blanks=1 ./guide.sgml
	    else
		echo "- ++ Warning: groff not available, cannot build \"${docformat}\" format." >&2
	    fi
	    ;;
	pdf)
	    echo "- Building pdf docs" >&2
	    $TMPDIR/linuxdoc --backend=latex --output=pdf \
			     --pass="\usepackage{times}" ./guide.sgml
	    ;;
	info)
	    echo "- Building info docs" >&2
	    $TMPDIR/linuxdoc --backend=info ./guide.sgml
	    ;;
	lyx)
	    echo "- Building lyx docs" >&2
	    $TMPDIR/linuxdoc --backend=lyx ./guide.sgml
	    ;;
	html)
	    echo "- Building html docs" >&2
	    $TMPDIR/linuxdoc --imagebuttons --backend=html ./guide.sgml \
		&& mv -f ./guide*.html ./html
	    ;;
	rtf)
	    echo "- Building rtf docs" >&2
	    $TMPDIR/linuxdoc --backend=rtf ./guide.sgml && if [ ! -d ./rtf ]; \
		then mkdir -m 755 ./rtf; fi && mv -f ./guide*.rtf ./rtf
	    ;;
	dvi+ps)
	    echo "- Building latex docs" >&2
	    if [ -n "`which latex`" ]; then
		echo "- Building dvi docs" >&2
		$TMPDIR/linuxdoc --backend=latex --output=dvi \
				 --pass="\usepackage{times}" ./guide.sgml

		if [ -n "`which dvips`" ]; then
		    echo "   + dvips" >&2
		    DVIPS_PAPER="a4"
		    if [ -r /etc/papersize ]; then
			TMP_PAPER=`head -n 1 /etc/papersize`
			if [ "${TMP_PAPER}" = "letter" ]; then
			    DVIPS_PAPER="letter"
			fi
		    fi
		    dvips -t ${DVIPS_PAPER} -o ./guide.ps ./guide.dvi
		    if [ -n "`which gzip`" -a -f ./guide.ps ]; then
			gzip -fn ./guide.ps
		    fi
		else
		    echo "- ++ Warning: dvips not available, cannot build \"guide.ps\"." >&2
		fi
	    else
		echo "- ++ Warning: latex not available, cannot build \"${docformat}\" format." >&2
	    fi
	    ;;
	sgml)
	    echo "- No need to build already present linuxdoc sgml." >&2
	    ;;
	*)
	    echo "- ++ Warning: Ignoring unknown \"${docformat}\" format." >&2
    esac
done

# Remove temporary directory.
rm -rf $TMPDIR

exit 0
